import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { 
  Bath, 
  CheckCircle, 
  Phone, 
  Mail,
  Users,
  Ruler,
  Droplets,
  Shield,
  Wrench,
  Building
} from "lucide-react";
import Link from "next/link";

export default function ToiletUnitsPage() {
  const toiletModels = [
    {
      name: "Single Toilet Unit",
      description: "Compact single toilet solution for basic needs",
      dimensions: "1.2m x 1.2m x 2.2m (H)",
      capacity: "Single user",
      features: ["Basic WC", "Ventilation", "Easy Clean", "Lockable Door", "Water Ready"]
    },
    {
      name: "Toilet with Washbasin",
      description: "Complete sanitation unit with washing facility",
      dimensions: "1.5m x 1.2m x 2.2m (H)",
      capacity: "Single user",
      features: ["WC + Washbasin", "Mirror", "Soap Dispenser", "Hand Dryer Ready", "Tile Finish"]
    },
    {
      name: "Multiple Toilet Block",
      description: "Multi-unit toilet block for high traffic areas",
      dimensions: "Customizable up to 20 units",
      capacity: "Multiple users",
      features: ["Multiple Cubicles", "Common Wash Area", "Disabled Access", "Vandal Resistant", "Easy Maintenance"]
    }
  ];

  const sanitationFeatures = [
    {
      icon: <Droplets className="h-6 w-6" />,
      title: "Hygienic Design",
      description: "Easy-to-clean surfaces with antimicrobial coatings and proper drainage"
    },
    {
      icon: <Shield className="h-6 w-6" />,
      title: "Durable Construction",
      description: "Weather-resistant materials built to withstand heavy usage"
    },
    {
      icon: <Wrench className="h-6 w-6" />,
      title: "Easy Maintenance",
      description: "Accessible plumbing and simple cleaning procedures"
    },
    {
      icon: <Building className="h-6 w-6" />,
      title: "Modular System",
      description: "Expandable units that can be connected for larger facilities"
    }
  ];

  const specifications = [
    "High-quality sanitary fittings",
    "Proper ventilation system",
    "Non-slip flooring material",
    "Lockable doors with occupancy indicators",
    "Water supply and drainage connections",
    "LED lighting with motion sensors",
    "Toilet paper and soap dispensers",
    "Hand washing facilities",
    "Privacy partitions and walls",
    "Emergency access capability"
  ];

  const applications = [
    "Construction sites",
    "Events and exhibitions",
    "Outdoor festivals",
    "Sports facilities",
    "Industrial facilities",
    "Schools and institutions",
    "Public parks and beaches",
    "Emergency and disaster relief",
    "Temporary installations",
    "Remote work locations"
  ];

  const configurations = [
    {
      units: "1-3 Units",
      description: "Small installations for limited users",
      suitable: "Small construction sites, security posts"
    },
    {
      units: "4-10 Units", 
      description: "Medium installations for regular traffic",
      suitable: "Medium construction projects, offices"
    },
    {
      units: "11-20 Units",
      description: "Large installations for high traffic",
      suitable: "Large projects, events, industrial sites"
    }
  ];

  return (
    <div className="min-h-screen">
      {/* Hero Section */}
      <section className="relative py-20 bg-gradient-to-r from-blue-900/80 to-gray-900/80">
        <div className="absolute inset-0 bg-black/40"></div>
        <div className="absolute inset-0">
          <div className="w-full h-full bg-gradient-to-br from-blue-800 via-gray-800 to-slate-800"></div>
        </div>
        
        <div className="relative z-20 container mx-auto px-4 text-center text-white">
          <h1 className="text-4xl md:text-6xl font-bold mb-6">
            Toilet Units
          </h1>
          <p className="text-xl md:text-2xl max-w-3xl mx-auto">
            Hygienic Portable Sanitation Solutions
          </p>
        </div>
      </section>

      {/* Product Overview */}
      <section className="py-16 bg-white">
        <div className="container mx-auto px-4">
          <div className="max-w-4xl mx-auto text-center mb-16">
            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-6">
              Complete Sanitation Solutions
            </h2>
            <p className="text-lg text-gray-700 leading-relaxed mb-6">
              Our porta cabin toilet units provide clean, hygienic sanitation facilities from single units 
              to customized installations up to 20 units. Designed for durability and easy maintenance 
              across various applications in UAE.
            </p>
          </div>

          {/* Toilet Models Grid */}
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8 mb-16">
            {toiletModels.map((model, index) => (
              <Card key={index} className="overflow-hidden hover:shadow-lg transition-shadow">
                <div className="relative h-48 bg-gradient-to-br from-cyan-800 to-blue-800">
                  <Badge className="absolute top-4 left-4 bg-cyan-600">
                    Sanitation
                  </Badge>
                </div>
                
                <CardHeader>
                  <CardTitle className="text-xl font-bold">{model.name}</CardTitle>
                  <CardDescription>{model.description}</CardDescription>
                </CardHeader>
                
                <CardContent>
                  <div className="space-y-3 mb-4">
                    <div className="flex items-center text-sm text-gray-600">
                      <Ruler className="h-4 w-4 mr-2" />
                      <span>{model.dimensions}</span>
                    </div>
                    <div className="flex items-center text-sm text-gray-600">
                      <Users className="h-4 w-4 mr-2" />
                      <span>{model.capacity}</span>
                    </div>
                  </div>

                  <div className="mb-4">
                    <h4 className="text-sm font-semibold text-gray-900 mb-2">Features:</h4>
                    <div className="space-y-1">
                      {model.features.slice(0, 3).map((feature, idx) => (
                        <div key={idx} className="flex items-center text-sm text-gray-600">
                          <CheckCircle className="h-3 w-3 mr-2 text-cyan-600" />
                          <span>{feature}</span>
                        </div>
                      ))}
                    </div>
                  </div>
                  
                  <Button className="w-full bg-cyan-600 hover:bg-cyan-700">
                    <Phone className="mr-2 h-4 w-4" />
                    Get Quote
                  </Button>
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Sanitation Features */}
          <div className="mb-16">
            <h2 className="text-3xl font-bold text-gray-900 text-center mb-12">
              Quality & Hygiene Features
            </h2>
            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
              {sanitationFeatures.map((feature, index) => (
                <div key={index} className="text-center">
                  <div className="bg-cyan-100 w-16 h-16 rounded-full flex items-center justify-center mx-auto mb-4">
                    <div className="text-cyan-600">
                      {feature.icon}
                    </div>
                  </div>
                  <h3 className="text-lg font-semibold text-gray-900 mb-2">
                    {feature.title}
                  </h3>
                  <p className="text-gray-600 text-sm">
                    {feature.description}
                  </p>
                </div>
              ))}
            </div>
          </div>
        </div>
      </section>

      {/* Configurations */}
      <section className="py-16 bg-gray-50">
        <div className="container mx-auto px-4">
          <h2 className="text-3xl font-bold text-gray-900 text-center mb-12">
            Available Configurations
          </h2>
          
          <div className="grid grid-cols-1 md:grid-cols-3 gap-8 mb-12">
            {configurations.map((config, index) => (
              <Card key={index} className="text-center p-6 hover:shadow-lg transition-shadow">
                <CardHeader className="pb-4">
                  <CardTitle className="text-2xl font-bold text-cyan-600">{config.units}</CardTitle>
                  <CardDescription className="text-lg">{config.description}</CardDescription>
                </CardHeader>
                <CardContent>
                  <p className="text-gray-600 text-sm">
                    <span className="font-medium">Suitable for:</span> {config.suitable}
                  </p>
                </CardContent>
              </Card>
            ))}
          </div>

          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12">
            {/* Specifications */}
            <div>
              <h3 className="text-2xl font-bold text-gray-900 mb-6 flex items-center">
                <Bath className="h-7 w-7 mr-3 text-cyan-600" />
                Technical Specifications
              </h3>
              <div className="space-y-3">
                {specifications.map((spec, index) => (
                  <div key={index} className="flex items-start space-x-3">
                    <CheckCircle className="h-5 w-5 text-cyan-600 mt-0.5 flex-shrink-0" />
                    <span className="text-gray-700">{spec}</span>
                  </div>
                ))}
              </div>
            </div>

            {/* Applications */}
            <div>
              <h3 className="text-2xl font-bold text-gray-900 mb-6 flex items-center">
                <Building className="h-7 w-7 mr-3 text-green-600" />
                Applications
              </h3>
              <div className="grid grid-cols-1 sm:grid-cols-2 gap-3">
                {applications.map((application, index) => (
                  <div key={index} className="flex items-center space-x-3 p-2">
                    <div className="w-2 h-2 bg-green-600 rounded-full"></div>
                    <span className="text-gray-700">{application}</span>
                  </div>
                ))}
              </div>
            </div>
          </div>
        </div>
      </section>

      {/* Call to Action */}
      <section className="py-16 bg-cyan-800 text-white">
        <div className="container mx-auto px-4 text-center">
          <h2 className="text-3xl md:text-4xl font-bold mb-4">
            Need Sanitation Solutions?
          </h2>
          <p className="text-xl mb-8 max-w-2xl mx-auto">
            From single toilet units to large multi-unit installations, we provide 
            complete sanitation solutions. Contact us for customized toilet facilities.
          </p>
          <div className="flex flex-col sm:flex-row gap-4 justify-center">
            <Button size="lg" className="bg-white text-cyan-800 hover:bg-gray-100 px-8">
              <Phone className="mr-2 h-5 w-5" />
              Call: +971 58 201 2073
            </Button>
            <Button size="lg" variant="outline" className="border-white text-white hover:bg-white/10 px-8" asChild>
              <Link href="/contact">
                <Mail className="mr-2 h-5 w-5" />
                Request Quote
              </Link>
            </Button>
          </div>
        </div>
      </section>
    </div>
  );
}